/// @file logger.h
///
/// @brief Describes the logger base class
///
/// @component Uspi/DeviceDetector
///
/// @author F.Berat / ADITG/SWG / fberat@de.adit-jv.com
///
/// @copyright (c) 2016 Advanced Driver Information Technology.
/// This code is developed by Advanced Driver Information Technology.
/// Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
/// All rights reserved.
///
/// @see DltLog SyslogLogger

#ifndef __DD_LOGGER_H__
#define __DD_LOGGER_H__

#include <iostream>
#include <string>
#include <sys/syslog.h>

namespace adit {
namespace uspi {

#ifndef LOGGER_IDENTITY
#define LOGGER_IDENTITY "" ///< Default identity to be used by the logger
#endif

/// @var ddlog
extern std::ostream ddlog;

/// @enum LogPriority
/// Generic definitions for log priorities, based on syslog priorities.
enum LogPriority {
    kLogEmerg   = LOG_EMERG,   ///< System is unusable
    kLogAlert   = LOG_ALERT,   ///< Action must be taken immediately
    kLogCrit    = LOG_CRIT,    ///< critical conditions
    kLogErr     = LOG_ERR,     ///< Error conditions
    kLogWarning = LOG_WARNING, ///< Warning conditions
    kLogNotice  = LOG_NOTICE,  ///< Normal, but significant, condition
    kLogInfo    = LOG_INFO,    ///< Informational message
    kLogDebug   = LOG_DEBUG    ///< Debug-level message
};

/// New operator<< definition for the Log class. Defined in @ref logger.cpp.
std::ostream& operator<< (std::ostream& os, const LogPriority& log_priority);

/// @brief The base logger class
///
/// Allows creation of subclass logger using non-standard-file based output
class Log : public std::basic_streambuf<char, std::char_traits<char> > {
public:
    virtual ~Log() {};

protected:
    /// @brief Flushes the @ref mBuffer
    virtual int sync() = 0;
    /// @brief Add a character to the buffer and sync if EOF is met.
    /// @param c The character to push into the @ref mBuffer
    virtual int overflow(int c) = 0;

    friend std::ostream& operator<< (std::ostream& os,
                                     const LogPriority& log_priority);
    /// The temporary buffer used to append logs until EOF character
    std::string mBuffer;
    /// Current log priority
    LogPriority mPriority;
    /// Default log priority, to reinitialize @ref mPriority
    LogPriority mDefaultPriority;
    /// Identity of the logger
    std::string mIdent;
};

} // namespace uspi
} // namespace adit
#endif // __DD_LOGGER_H__
